<?php
/*
Plugin Name: AI Bulk ChatGPT Silo
Description: Tạo bài viết hàng loạt từ danh sách từ khóa bằng API ChatGPT (queue nền, đa ngôn ngữ, license serial + email + domain, quản lý số bài viết, tạo link nội bộ silo).
Version: 3.2
Author: Công ty TNHH Fame Media
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AI_Bulk_ChatGPT_Silo {

    const OPTION_SETTINGS = 'abcgs_settings';
    const OPTION_QUEUE    = 'abcgs_queue';
    const OPTION_SILO_LOG = 'abcgs_silo_log';
    const OPTION_LICENSE  = 'abcgs_license';

    const CRON_HOOK       = 'abcgs_process_queue_event';

    public function __construct() {
        @set_time_limit( 300 );
        @ini_set( 'max_execution_time', 300 );

        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( self::CRON_HOOK, array( $this, 'process_queue' ) );
    }

    public function add_admin_menu() {
        add_menu_page(
            'AI Bulk ChatGPT Silo',
            'AI Bulk Silo',
            'manage_options',
            'abcgs-ai-bulk-silo',
            array( $this, 'render_admin_page' ),
            'dashicons-edit',
            27
        );
    }

    public function render_admin_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( isset( $_POST['abcgs_start'] ) ) {
            $this->handle_start_queue();
        }

        $settings = get_option( self::OPTION_SETTINGS, array(
            'api_key'         => '',
            'model'           => 'gpt-4.1-mini',
            'category_id'     => 0,
            'keyword_density' => 5,
            'post_status'     => 'publish',
            'post_length'     => 1200,
            'language'        => 'vi',
        ) );

        $api_key         = $settings['api_key'] ?? '';
        $model           = $settings['model'] ?? 'gpt-4.1-mini';
        $category_id     = intval( $settings['category_id'] ?? 0 );
        $keyword_density = floatval( $settings['keyword_density'] ?? 5 );
        $post_status     = $settings['post_status'] ?? 'publish';
        $post_length     = intval( $settings['post_length'] ?? 1200 );
        $language        = $settings['language'] ?? 'vi';

        $available_models = array(
            'gpt-4.1-mini'  => 'gpt-4.1-mini (nhanh, tiet kiem)',
            'gpt-4.1'       => 'gpt-4.1 (chat luong cao)',
            'gpt-4o-mini'   => 'gpt-4o-mini (toi uu chi phi)',
            'gpt-4o'        => 'gpt-4o (da nang cao cap)',
            'o3-mini'       => 'o3-mini (ly luan manh)',
            'gpt-3.5-turbo' => 'gpt-3.5-turbo (doi cu, gia re)',
        );

        $model_is_known = array_key_exists( $model, $available_models );
        $custom_model   = $model_is_known ? '' : $model;

        $queue = get_option( self::OPTION_QUEUE, array() );
        $queue_size = is_array( $queue ) ? count( $queue ) : 0;

        $silo_log = get_option( self::OPTION_SILO_LOG, '' );

        $license = get_option( self::OPTION_LICENSE, array(
            'serial'     => '',
            'email'      => '',
            'status'     => 'invalid',
            'max_posts'  => 0,
            'used_posts' => 0,
        ) );

        $max_posts  = intval( $license['max_posts'] ?? 0 );
        $used_posts = intval( $license['used_posts'] ?? 0 );
        $remain     = max( 0, $max_posts - $used_posts );
        ?>

        <div class="wrap">
            <h1>AI Bulk ChatGPT Silo</h1>

            <?php if ( isset( $_GET['abcgs_message'] ) ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html( wp_unslash( $_GET['abcgs_message'] ) ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( isset( $_GET['abcgs_error'] ) ) : ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html( wp_unslash( $_GET['abcgs_error'] ) ); ?></p>
                </div>
            <?php endif; ?>

            <form method="post">
                <?php wp_nonce_field( 'abcgs_start_queue', 'abcgs_nonce' ); ?>

                <h2>License / Serial key</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_serial_key">Seri key</label></th>
                        <td>
                            <input type="text"
                                   name="abcgs_serial_key"
                                   id="abcgs_serial_key"
                                   value="<?php echo esc_attr( $license['serial'] ?? '' ); ?>"
                                   class="regular-text"
                                   placeholder="VD: BASIC-XXXX-YYYY">
                            <br><br>
                            <label for="abcgs_license_email"><strong>Email license</strong></label><br>
                            <input type="email"
                                   name="abcgs_license_email"
                                   id="abcgs_license_email"
                                   value="<?php echo esc_attr( $license['email'] ?? '' ); ?>"
                                   class="regular-text"
                                   placeholder="Email dung khi mua plugin">

                            <?php
                            $status_text  = 'Chua kich hoat';
                            $status_class = 'notice-warning';
                            if ( isset( $license['status'] ) && $license['status'] === 'valid' ) {
                                $status_text  = 'License hop le';
                                $status_class = 'notice-success';
                            } elseif ( isset( $license['status'] ) && $license['status'] === 'invalid' && ! empty( $license['serial'] ) ) {
                                $status_text  = 'License khong hop le hoac da het han';
                                $status_class = 'notice-error';
                            }
                            ?>
                            <div class="notice <?php echo esc_attr( $status_class ); ?>" style="margin-top:10px;padding:8px;">
                                <p>
                                    Trang thai license: <strong><?php echo esc_html( $status_text ); ?></strong><br>
                                    Email license: <strong><?php echo esc_html( $license['email'] ?? '' ); ?></strong><br>
                                    So bai da tao (server): <strong><?php echo esc_html( $used_posts ); ?></strong><br>
                                    Gioi han toi da: <strong><?php echo esc_html( $max_posts ); ?></strong><br>
                                    Con lai (server): <strong><?php echo esc_html( $remain ); ?></strong>
                                </p>
                            </div>
                            <p class="description">
                                Moi license gan voi 1 email + domain + gioi han so bai (500 / 800 / 1000...).
                            </p>
                        </td>
                    </tr>
                </table>

                <h2>Cau hinh API</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_api_key">OpenAI API key</label></th>
                        <td>
                            <input type="password"
                                   name="abcgs_api_key"
                                   id="abcgs_api_key"
                                   value="<?php echo esc_attr( $api_key ); ?>"
                                   class="regular-text"
                                   placeholder="sk-...">
                            <p class="description">Dan API key OpenAI (ChatGPT API).</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_model_select">Model ChatGPT</label></th>
                        <td>
                            <select name="abcgs_model_select" id="abcgs_model_select">
                                <?php foreach ( $available_models as $value => $label ) : ?>
                                    <option value="<?php echo esc_attr( $value ); ?>" <?php selected( $model_is_known ? $model : '', $value ); ?>>
                                        <?php echo esc_html( $label ); ?>
                                    </option>
                                <?php endforeach; ?>
                                <option value="custom" <?php selected( ! $model_is_known ); ?>>Model khac (nhap tay)</option>
                            </select>
                            <p style="margin-top:8px;">
                                <label for="abcgs_model_custom"><strong>Model tuy chinh:</strong></label><br>
                                <input type="text"
                                       name="abcgs_model_custom"
                                       id="abcgs_model_custom"
                                       value="<?php echo esc_attr( $custom_model ); ?>"
                                       class="regular-text"
                                       placeholder="VD: gpt-4.1-mini">
                            </p>
                            <p class="description">Neu chon "Model khac" thi se dung model trong o nay.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_language">Ngon ngu bai viet</label></th>
                        <td>
                            <select name="abcgs_language" id="abcgs_language">
                                <option value="vi" <?php selected( $language, 'vi' ); ?>>Tieng Viet</option>
                                <option value="en" <?php selected( $language, 'en' ); ?>>Tieng Anh</option>
                                <option value="zh" <?php selected( $language, 'zh' ); ?>>Tieng Trung (gian the)</option>
                            </select>
                            <p class="description">Chon ngon ngu bai viet duoc sinh ra.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_category_id">Chuyen muc</label></th>
                        <td>
                            <?php
                            wp_dropdown_categories( array(
                                'show_option_all' => '— Chon chuyen muc —',
                                'hide_empty'      => false,
                                'name'            => 'abcgs_category_id',
                                'id'              => 'abcgs_category_id',
                                'selected'        => $category_id,
                                'taxonomy'        => 'category',
                            ) );
                            ?>
                            <p class="description">Tat ca bai viet se dang vao chuyen muc nay.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_post_status">Trang thai bai viet</label></th>
                        <td>
                            <select name="abcgs_post_status" id="abcgs_post_status">
                                <option value="publish" <?php selected( $post_status, 'publish' ); ?>>Dang ngay (publish)</option>
                                <option value="draft" <?php selected( $post_status, 'draft' ); ?>>Luu nhap (draft)</option>
                            </select>
                        </td>
                    </tr>
                </table>

                <h2>Tuy chon noi dung</h2>
                <table class="form-table">
                    <tr>
                        <th><label for="abcgs_post_length">Do dai bai viet (so tu)</label></th>
                        <td>
                            <input type="number"
                                   name="abcgs_post_length"
                                   id="abcgs_post_length"
                                   value="<?php echo esc_attr( $post_length ); ?>"
                                   min="200"
                                   step="100">
                            <p class="description">VD: 800, 1200, 2000...</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="abcgs_keyword_density">Mat do tu khoa (%)</label></th>
                        <td>
                            <input type="number"
                                   name="abcgs_keyword_density"
                                   id="abcgs_keyword_density"
                                   value="<?php echo esc_attr( $keyword_density ); ?>"
                                   min="1"
                                   max="20"
                                   step="0.5">
                            <p class="description">VD: 3, 3.5, 5 (%).</p>
                        </td>
                    </tr>
                </table>

                <h2>Danh sach tu khoa</h2>
                <p>
                    Moi dong la <strong>mot tu khoa</strong>.<br>
                    Dong 1 = <strong>silo page</strong> (bai tru cot).<br>
                    Cac dong sau = bai silo con.<br>
                    Plugin se tu tao <strong>link noi bo silo</strong> khi hoan thanh.
                </p>
                <textarea name="abcgs_keywords"
                          rows="10"
                          style="width:100%;max-width:800px;"
                          placeholder="Vi du:
Cambodia restaurant guide
Best Cambodian seafood restaurant
Authentic Cambodian street food
Fine dining in Phnom Penh"></textarea>

                <p class="submit">
                    <button type="submit" name="abcgs_start" class="button button-primary button-large">
                        Bat dau xep hang tao bai (Queue nen)
                    </button>
                </p>
            </form>

            <hr>

            <h2>Trang thai queue</h2>
            <p>
                So muc trong queue: <strong><?php echo intval( $queue_size ); ?></strong><br>
                <?php
                $pending = 0;
                $done    = 0;
                if ( $queue_size > 0 ) {
                    foreach ( $queue as $item ) {
                        if ( isset( $item['status'] ) && $item['status'] === 'done' ) {
                            $done++;
                        } else {
                            $pending++;
                        }
                    }
                }
                ?>
                Con dang xu ly: <strong><?php echo intval( $pending ); ?></strong><br>
                Da xong: <strong><?php echo intval( $done ); ?></strong>
            </p>

            <?php if ( ! empty( $silo_log ) ) : ?>
                <p><strong>Trang thai silo:</strong> <?php echo esc_html( $silo_log ); ?></p>
            <?php endif; ?>

            <?php if ( $queue_size > 0 ) : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Tu khoa</th>
                            <th>ID bai viet</th>
                            <th>Trang thai</th>
                            <th>Ghi chu</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php
                    $i = 1;
                    foreach ( $queue as $item ) :
                        ?>
                        <tr>
                            <td><?php echo intval( $i ); ?></td>
                            <td><?php echo esc_html( $item['keyword'] ?? '' ); ?></td>
                            <td><?php echo ! empty( $item['post_id'] ) ? intval( $item['post_id'] ) : '-'; ?></td>
                            <td><?php echo esc_html( $item['status'] ?? '' ); ?></td>
                            <td><?php echo esc_html( $item['content_status'] ?? '' ); ?></td>
                        </tr>
                        <?php
                        $i++;
                    endforeach;
                    ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p>Chua co queue nao hoac da xu ly xong.</p>
            <?php endif; ?>
        </div>

        <?php
    }

    private function handle_start_queue() {
        if ( ! isset( $_POST['abcgs_nonce'] ) || ! wp_verify_nonce( $_POST['abcgs_nonce'], 'abcgs_start_queue' ) ) {
            wp_die( 'Xac thuc khong hop le.' );
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Khong co quyen.' );
        }

        $api_key          = isset( $_POST['abcgs_api_key'] ) ? sanitize_text_field( $_POST['abcgs_api_key'] ) : '';
        $model_select     = isset( $_POST['abcgs_model_select'] ) ? sanitize_text_field( $_POST['abcgs_model_select'] ) : 'gpt-4.1-mini';
        $model_custom_raw = isset( $_POST['abcgs_model_custom'] ) ? sanitize_text_field( $_POST['abcgs_model_custom'] ) : '';
        $category_id      = isset( $_POST['abcgs_category_id'] ) ? intval( $_POST['abcgs_category_id'] ) : 0;
        $post_status      = isset( $_POST['abcgs_post_status'] ) ? sanitize_text_field( $_POST['abcgs_post_status'] ) : 'publish';
        $post_length      = isset( $_POST['abcgs_post_length'] ) ? intval( $_POST['abcgs_post_length'] ) : 1200;
        $keyword_density  = isset( $_POST['abcgs_keyword_density'] ) ? floatval( $_POST['abcgs_keyword_density'] ) : 5;
        $language         = isset( $_POST['abcgs_language'] ) ? sanitize_text_field( $_POST['abcgs_language'] ) : 'vi';
        $serial_key       = isset( $_POST['abcgs_serial_key'] ) ? sanitize_text_field( $_POST['abcgs_serial_key'] ) : '';
        $license_email    = isset( $_POST['abcgs_license_email'] ) ? sanitize_email( $_POST['abcgs_license_email'] ) : '';

        if ( $model_select === 'custom' ) {
            $model = ! empty( $model_custom_raw ) ? $model_custom_raw : 'gpt-4.1-mini';
        } else {
            $model = $model_select;
        }

        if ( empty( $api_key ) || strlen( $api_key ) < 20 ) {
            $this->redirect_with_error( 'API key trong hoac khong hop le.' );
        }

        if ( $category_id <= 0 ) {
            $this->redirect_with_error( 'Vui long chon chuyen muc.' );
        }

        if ( $post_length < 200 ) {
            $post_length = 200;
        }

        if ( $keyword_density <= 0 ) {
            $keyword_density = 3;
        }

        if ( ! in_array( $post_status, array( 'publish', 'draft' ), true ) ) {
            $post_status = 'publish';
        }

        if ( ! in_array( $language, array( 'vi', 'en', 'zh' ), true ) ) {
            $language = 'vi';
        }

        if ( empty( $_POST['abcgs_keywords'] ) ) {
            $this->redirect_with_error( 'Vui long nhap danh sach tu khoa.' );
        }

        if ( empty( $serial_key ) ) {
            $this->redirect_with_error( 'Vui long nhap seri key.' );
        }

        if ( empty( $license_email ) ) {
            $this->redirect_with_error( 'Vui long nhap email license.' );
        }

        $raw_keywords = wp_unslash( $_POST['abcgs_keywords'] );
        $lines        = preg_split( '/\r\n|\r|\n/', $raw_keywords );
        $keywords     = array();

        foreach ( $lines as $line ) {
            $line = trim( $line );
            if ( ! empty( $line ) ) {
                $keywords[] = $line;
            }
        }

        if ( empty( $keywords ) ) {
            $this->redirect_with_error( 'Danh sach tu khoa khong hop le.' );
        }

        // Validate license qua server
        $license_info  = array();
        $license_error = '';

        if ( ! $this->validate_license_key( $serial_key, $license_email, $license_info, $license_error ) ) {
            update_option( self::OPTION_LICENSE, array(
                'serial'     => $serial_key,
                'email'      => $license_email,
                'status'     => 'invalid',
                'max_posts'  => 0,
                'used_posts' => 0,
            ) );
            $this->redirect_with_error( 'License khong hop le: ' . $license_error );
        }

        $license_old = get_option( self::OPTION_LICENSE, array(
            'serial'     => '',
            'email'      => '',
            'status'     => 'invalid',
            'max_posts'  => 0,
            'used_posts' => 0,
        ) );

        // Lay used_posts tu server (neu co)
        $used_posts = intval( $license_info['used_posts'] ?? 0 );
        $max_posts  = intval( $license_info['max_posts'] ?? 0 );

        if (
            ! empty( $license_old['serial'] ) &&
            strtoupper( $license_old['serial'] ) === strtoupper( $serial_key ) &&
            ! empty( $license_old['email'] ) &&
            strtolower( $license_old['email'] ) === strtolower( $license_email )
        ) {
            // Neu local cao hon server (truong hop sync), uu tien so lon hon de an toan
            $used_posts = max( $used_posts, intval( $license_old['used_posts'] ?? 0 ) );
        }

        if ( $max_posts <= 0 ) {
            $this->redirect_with_error( 'License valid nhung max_posts <= 0.' );
        }

        if ( $used_posts >= $max_posts ) {
            update_option( self::OPTION_LICENSE, array(
                'serial'     => $serial_key,
                'email'      => $license_email,
                'status'     => 'valid',
                'max_posts'  => $max_posts,
                'used_posts' => $used_posts,
            ) );
            $this->redirect_with_error( 'Ban da dung het so bai cho license nay.' );
        }

        update_option( self::OPTION_LICENSE, array(
            'serial'     => $serial_key,
            'email'      => $license_email,
            'status'     => 'valid',
            'max_posts'  => $max_posts,
            'used_posts' => $used_posts,
        ) );

        update_option( self::OPTION_SETTINGS, array(
            'api_key'         => $api_key,
            'model'           => $model,
            'category_id'     => $category_id,
            'keyword_density' => $keyword_density,
            'post_status'     => $post_status,
            'post_length'     => $post_length,
            'language'        => $language,
        ) );

        $queue = array();
        foreach ( $keywords as $kw ) {
            $queue[] = array(
                'keyword'        => $kw,
                'post_id'        => 0,
                'status'         => 'pending',
                'content_status' => '',
            );
        }

        update_option( self::OPTION_QUEUE, $queue );
        update_option( self::OPTION_SILO_LOG, '' );

        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            wp_schedule_single_event( time() + 10, self::CRON_HOOK );
        }

        $this->redirect_with_message( 'Da xep hang ' . count( $queue ) . ' tu khoa. Cron se lan luot tao bai va link silo.' );
    }

    public function process_queue() {
        $queue = get_option( self::OPTION_QUEUE, array() );
        if ( empty( $queue ) || ! is_array( $queue ) ) {
            return;
        }

        $settings = get_option( self::OPTION_SETTINGS, array() );
        $api_key  = $settings['api_key'] ?? '';
        $model    = $settings['model'] ?? 'gpt-4.1-mini';

        if ( empty( $api_key ) || strlen( $api_key ) < 20 ) {
            return;
        }

        $category_id     = intval( $settings['category_id'] ?? 0 );
        $post_status     = $settings['post_status'] ?? 'publish';
        $post_length     = intval( $settings['post_length'] ?? 1200 );
        $keyword_density = floatval( $settings['keyword_density'] ?? 5 );
        $language        = $settings['language'] ?? 'vi';

        $license = get_option( self::OPTION_LICENSE, array(
            'serial'     => '',
            'email'      => '',
            'status'     => 'invalid',
            'max_posts'  => 0,
            'used_posts' => 0,
        ) );

        $serial_valid = ( isset( $license['status'] ) && $license['status'] === 'valid' );
        $max_posts    = intval( $license['max_posts'] ?? 0 );
        $used_posts   = intval( $license['used_posts'] ?? 0 );

        $batch_size = 2; // moi lan cron chi xu ly toi da 2 bai
        $processed  = 0;

        foreach ( $queue as $index => $item ) {
            if ( $processed >= $batch_size ) {
                break;
            }

            if ( isset( $item['status'] ) && $item['status'] === 'done' ) {
                continue;
            }

            $keyword = $item['keyword'] ?? '';
            if ( empty( $keyword ) ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Bo qua: tu khoa rong.';
                continue;
            }

            if ( ! $serial_valid ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Dung: license khong hop le hoac chua kich hoat.';
                continue;
            }

            if ( $used_posts >= $max_posts ) {
                $queue[ $index ]['status']         = 'done';
                $queue[ $index ]['content_status'] = 'Dung: da het so bai theo license.';
                continue;
            }

            if ( empty( $item['post_id'] ) ) {

                $prompt  = $this->build_prompt_for_keyword( $keyword, $post_length, $keyword_density, $language );
                $content = $this->call_chatgpt( $api_key, $model, $prompt, $error_message_content );

                if ( ! $content ) {
                    $queue[ $index ]['content_status'] = 'Loi API: ' . $error_message_content;
                    $queue[ $index ]['status']         = 'done';
                    $processed++;
                    continue;
                }

                if ( stripos( $content, '<h1' ) === false ) {
                    $queue[ $index ]['content_status'] = 'Canh bao: khong tim thay H1.';
                } else {
                    $queue[ $index ]['content_status'] = 'OK: da nhan noi dung AI.';
                }

                $post_title = $this->extract_title_from_content( $content, $keyword );
                $excerpt    = $this->generate_excerpt_from_html( $content, 40 );

                $post_id = wp_insert_post( array(
                    'post_title'    => $post_title,
                    'post_content'  => $content,
                    'post_status'   => in_array( $post_status, array( 'publish', 'draft' ), true ) ? $post_status : 'publish',
                    'post_author'   => get_current_user_id(),
                    'post_category' => $category_id > 0 ? array( $category_id ) : array(),
                    'post_excerpt'  => $excerpt,
                ), true );

                if ( is_wp_error( $post_id ) || ! $post_id ) {
                    $queue[ $index ]['content_status'] .= ' | Loi: khong tao duoc post.';
                    $queue[ $index ]['status']          = 'done';
                    $processed++;
                    continue;
                }

                // Tao post thanh cong -> goi API consume de cap nhat used_posts tren server
                $license = $this->consume_remote_license_usage( $license, 1 );
                $used_posts = intval( $license['used_posts'] ?? $used_posts );
                $max_posts  = intval( $license['max_posts']  ?? $max_posts );

                update_option( self::OPTION_LICENSE, array(
                    'serial'     => $license['serial'] ?? '',
                    'email'      => $license['email'] ?? '',
                    'status'     => 'valid',
                    'max_posts'  => $max_posts,
                    'used_posts' => $used_posts,
                ) );

                $queue[ $index ]['post_id'] = $post_id;
            }

            $queue[ $index ]['status'] = 'done';
            $processed++;
        }

        update_option( self::OPTION_QUEUE, $queue );

        $pending_left = false;
        foreach ( $queue as $item ) {
            if ( $item['status'] !== 'done' ) {
                $pending_left = true;
                break;
            }
        }

        if ( $pending_left ) {
            wp_schedule_single_event( time() + 30, self::CRON_HOOK );
        } else {
            // Khi tat ca da xong -> tao link silo
            $post_ids = array();
            $keywords = array();
            foreach ( $queue as $item ) {
                if ( ! empty( $item['post_id'] ) ) {
                    $post_ids[] = intval( $item['post_id'] );
                    $keywords[] = $item['keyword'] ?? '';
                }
            }

            if ( count( $post_ids ) >= 2 ) {
                $silo_error = '';
                $silo_ok    = $this->apply_silo_internal_links( $post_ids, $keywords, $silo_error );
                if ( $silo_ok ) {
                    update_option( self::OPTION_SILO_LOG, 'OK: da gan link silo cho ' . count( $post_ids ) . ' bai.' );
                } else {
                    update_option( self::OPTION_SILO_LOG, 'Loi khi gan silo: ' . $silo_error );
                }
            } else {
                update_option( self::OPTION_SILO_LOG, 'Khong du bai de tao silo (toi thieu 2 bai).' );
            }
        }
    }

    private function build_prompt_for_keyword( $keyword, $post_length, $density, $language ) {
        $post_length     = max( 200, intval( $post_length ) );
        $keyword_density = floatval( $density );

        if ( $language === 'vi' ) {
            return "
Ban la mot copywriter SEO chuyen nghiep. Hay viet mot bai viet gioi thieu bang tieng Viet ve tu khoa chinh \"{$keyword}\" theo yeu cau sau:

1. Tieu de:
- Tu tao mot tieu de chinh KHONG trung hoan toan voi tu khoa chinh.
- Tieu de nam o DONG DAU TIEN cua bai viet.
- Tieu de co CHINH XAC 12 tu (dem theo khoang trang).

2. Do dai:
- Do dai noi dung khoang {$post_length} tu (co the chenhlech nhe).

3. Quy dinh bat buoc:
- KHONG su dung dau gach ngang '-' trong toan bo bai viet.
- KHONG neu gia san pham.
- KHONG ke cau chuyen ca nhan.
- KHONG dua ra bat ky cam ket hay dam bao nao.
- KHONG co muc Cau hoi thuong gap (FAQ).

4. Tu khoa:
- Tu khoa chinh: \"{$keyword}\".
- Mat do tu khoa chinh khoang {$keyword_density}% tren toan bai (khong can chinh xac tuyet doi, chi can hop ly cho SEO).
- Su dung them tu khoa phu, tu khoa lien quan, semantic keyword va tu khoa bong ma lien quan toi tu khoa chinh.
- Moi lan xuat hien tu khoa chinh \"{$keyword}\", in dam bang HTML: <strong>{$keyword}</strong>.

5. Dinh dang:
- Viet bang tieng Viet co dau, cau tu tu nhien, de doc.
- Chi su dung cac the HTML co ban: <h1>, <h2>, <h3>, <p>, <ul>, <li>.
- KHONG chen script, iframe hoac bat ky ma nhung nao khac.

Tra ve TOAN BO noi dung bai viet duoi dang HTML:
";
        }

        if ( $language === 'zh' ) {
            return "
You are an SEO copywriter. Write an informational article in Simplified Chinese about the main keyword \"{$keyword}\" with the following rules:

1. Title:
- Create a unique main title that is NOT exactly the same as the main keyword.
- The title must be on the FIRST line of the article.
- Wrap the title in an HTML <h1>...</h1> tag.
- The title must contain EXACTLY 12 segments separated by spaces.

2. Length:
- The article length should be around {$post_length} words (approximate is fine).

3. Strict rules:
- Do NOT use hyphen '-' in the article.
- Do NOT mention any product price.
- Do NOT tell personal stories.
- Do NOT make any promises or guarantees.
- Do NOT include any FAQ section.

4. Keywords:
- Main keyword: \"{$keyword}\".
- Keep the main keyword density around {$keyword_density}% (roughly).
- Use related keywords, semantic keywords, and latent keywords related to the main keyword.
- Every time the main keyword \"{$keyword}\" appears, wrap it with <strong>{$keyword}</strong>.

5. Formatting:
- Write in Simplified Chinese.
- Use only <h1>, <h2>, <h3>, <p>, <ul>, <li>.
- No scripts, no iframes, no embeds.

Return FULL HTML content only:
";
        }

        // English
        return "
You are an SEO copywriter. Write an informational article in English about the main keyword \"{$keyword}\" with the following rules:

1. Title:
- Create a unique main title that is NOT exactly the same as the main keyword.
- The title must be on the FIRST line of the article.
- Wrap the title in an HTML <h1>...</h1> tag.
- The title must contain EXACTLY 12 words.

2. Length:
- The article length should be around {$post_length} words (approximate is fine).

3. Strict rules:
- Do NOT use hyphen '-' in the article.
- Do NOT mention any product price.
- Do NOT tell personal stories.
- Do NOT make any promises or guarantees.
- Do NOT include any FAQ section.

4. Keywords:
- Main keyword: \"{$keyword}\".
- Keep the main keyword density around {$keyword_density}% (roughly).
- Use related keywords, semantic keywords, and ghost keywords related to the main keyword.
- Every time the main keyword \"{$keyword}\" appears, wrap it with <strong>{$keyword}</strong>.

5. Formatting:
- Use only <h1>, <h2>, <h3>, <p>, <ul>, <li>.
- No scripts, no iframes, no embeds.

Return FULL HTML content only:
";
    }

    private function call_chatgpt( $api_key, $model, $prompt, &$error_message = '' ) {
        $error_message = '';

        $response = wp_remote_post(
            'https://api.openai.com/v1/chat/completions',
            array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type'  => 'application/json',
                ),
                'body'    => wp_json_encode( array(
                    'model'    => $model,
                    'messages' => array(
                        array(
                            'role'    => 'system',
                            'content' => 'You are a strict SEO content writer. Always follow the user instructions exactly.',
                        ),
                        array(
                            'role'    => 'user',
                            'content' => $prompt,
                        ),
                    ),
                    'temperature' => 0.7,
                ) ),
                'timeout' => 60,
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $code < 200 || $code >= 300 ) {
            $error_message = 'HTTP ' . $code . ' | ' . substr( $body, 0, 300 );
            return false;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $error_message = 'JSON error: ' . json_last_error_msg();
            return false;
        }

        if ( empty( $data['choices'][0]['message']['content'] ) ) {
            $error_message = 'API khong tra ve content.';
            return false;
        }

        return $data['choices'][0]['message']['content'];
    }

    private function extract_title_from_content( $content, $fallback ) {
        $title = $fallback;

        if ( preg_match( '/<h1[^>]*>(.*?)<\/h1>/is', $content, $matches ) ) {
            $clean = wp_strip_all_tags( $matches[1] );
            $clean = trim( preg_replace( '/\s+/', ' ', $clean ) );
            if ( ! empty( $clean ) ) {
                $title = $clean;
            }
        }

        return $title;
    }

    private function generate_excerpt_from_html( $html, $word_limit = 40 ) {
        $text = wp_strip_all_tags( $html );
        $text = trim( preg_replace( '/\s+/', ' ', $text ) );
        if ( empty( $text ) ) {
            return '';
        }
        $words = explode( ' ', $text );
        if ( count( $words ) <= $word_limit ) {
            return $text;
        }
        return implode( ' ', array_slice( $words, 0, $word_limit ) ) . '...';
    }

    private function apply_silo_internal_links( $post_ids, $keywords, &$error_message = '' ) {
        $error_message = '';

        if ( empty( $post_ids ) || count( $post_ids ) < 2 ) {
            $error_message = 'Khong du bai de tao silo.';
            return false;
        }

        $count_posts    = count( $post_ids );
        $count_keywords = count( $keywords );
        $count          = min( $count_posts, $count_keywords );

        $silo_post_id = $post_ids[0];
        $silo_url     = get_permalink( $silo_post_id );
        $silo_anchor  = $keywords[0] ?? get_the_title( $silo_post_id );

        if ( ! $silo_url ) {
            $error_message = 'Khong lay duoc permalink silo page.';
            return false;
        }

        for ( $i = 0; $i < $count; $i++ ) {
            $post_id = $post_ids[ $i ];
            $content = get_post_field( 'post_content', $post_id );
            if ( empty( $content ) ) {
                continue;
            }

            $links = array();

            if ( $i === 0 ) {
                // Silo page: link toi bai ke tiep
                if ( $count > 1 ) {
                    $next_post_id = $post_ids[1];
                    $next_url     = get_permalink( $next_post_id );
                    $next_anchor  = $keywords[1] ?? get_the_title( $next_post_id );
                    if ( $next_url ) {
                        $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $next_url ), esc_html( $next_anchor ) );
                    }
                }
            } else {
                // Bai con: link toi bai tiep theo (neu co)
                if ( $i + 1 < $count ) {
                    $next_post_id = $post_ids[ $i + 1 ];
                    $next_url     = get_permalink( $next_post_id );
                    $next_anchor  = $keywords[ $i + 1 ] ?? get_the_title( $next_post_id );
                    if ( $next_url ) {
                        $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $next_url ), esc_html( $next_anchor ) );
                    }
                }
                // Va link ve silo page
                $links[] = sprintf( '<a href="%s">%s</a>', esc_url( $silo_url ), esc_html( $silo_anchor ) );
            }

            if ( empty( $links ) ) {
                continue;
            }

            $links_html = '<div class="silo-links"><p><strong>Lien ket noi bo cung chu de:</strong> ' .
                implode( ' | ', $links ) .
                '</p></div>';

            $new_content = $content . "\n\n" . $links_html;

            $updated = wp_update_post( array(
                'ID'           => $post_id,
                'post_content' => $new_content,
            ), true );

            if ( is_wp_error( $updated ) ) {
                $error_message = 'Loi update bai ' . $post_id . ': ' . $updated->get_error_message();
            }
        }

        return true;
    }

    private function validate_license_key( $serial_key, $license_email, &$license_info, &$error_message = '' ) {
        $error_message = '';
        $license_info  = array();

        $serial_key    = trim( $serial_key );
        $license_email = trim( $license_email );

        if ( empty( $serial_key ) ) {
            $error_message = 'Seri key trong.';
            return false;
        }
        if ( empty( $license_email ) ) {
            $error_message = 'Email license trong.';
            return false;
        }

        $site_url = site_url();
        $domain   = parse_url( $site_url, PHP_URL_HOST );
        if ( ! $domain ) {
            $domain = $site_url;
        }

        // SUA URL NAY THANH DOMAIN LICENSE SERVER CUA BAN
        $api_url = 'https://famemedia.vn/licenses/license-api.php';

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 20,
                'body'    => array(
                    'action'   => 'validate',
                    'serial'   => $serial_key,
                    'email'    => $license_email,
                    'domain'   => $domain,
                    'site_url' => $site_url,
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            $error_message = 'Khong ket noi duoc license server: ' . $response->get_error_message();
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $code < 200 || $code >= 300 ) {
            $error_message = 'License server HTTP ' . $code . ' | ' . substr( $body, 0, 200 );
            return false;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            $error_message = 'Khong parse duoc JSON: ' . json_last_error_msg();
            return false;
        }

        if ( empty( $data['status'] ) || $data['status'] !== 'ok' ) {
            $msg           = ! empty( $data['message'] ) ? $data['message'] : 'License invalid.';
            $error_message = 'License bi tu choi: ' . $msg;
            return false;
        }

        $max_posts  = isset( $data['max_posts'] ) ? intval( $data['max_posts'] ) : 0;
        $used_posts = isset( $data['used_posts'] ) ? intval( $data['used_posts'] ) : 0;

        if ( $max_posts <= 0 ) {
            $error_message = 'License valid nhung max_posts <= 0.';
            return false;
        }

        $plan = isset( $data['plan'] ) ? $data['plan'] : 'unknown';

        $license_info = array(
            'max_posts'  => $max_posts,
            'used_posts' => $used_posts,
            'plan'       => $plan,
        );

        return true;
    }

    private function consume_remote_license_usage( $license, $count = 1 ) {
        $serial = $license['serial'] ?? '';
        $email  = $license['email'] ?? '';

        if ( empty( $serial ) || empty( $email ) ) {
            return $license;
        }

        $site_url = site_url();
        $domain   = parse_url( $site_url, PHP_URL_HOST );
        if ( ! $domain ) {
            $domain = $site_url;
        }

        // SUA URL NAY THANH DOMAIN LICENSE SERVER CUA BAN
        $api_url = 'https://famemedia.vn/licenses/license-api.php';

        $response = wp_remote_post(
            $api_url,
            array(
                'timeout' => 20,
                'body'    => array(
                    'action'   => 'consume',
                    'serial'   => $serial,
                    'email'    => $email,
                    'domain'   => $domain,
                    'site_url' => $site_url,
                    'count'    => max( 1, intval( $count ) ),
                ),
            )
        );

        if ( is_wp_error( $response ) ) {
            return $license;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        if ( $code < 200 || $code >= 300 ) {
            return $license;
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            return $license;
        }

        if ( isset( $data['status'] ) && $data['status'] === 'ok' ) {
            $license['used_posts'] = intval( $data['used_posts'] ?? ( $license['used_posts'] ?? 0 ) );
            $license['max_posts']  = intval( $data['max_posts']  ?? ( $license['max_posts']  ?? 0 ) );
        }

        return $license;
    }

    private function redirect_with_message( $msg ) {
        $url = add_query_arg(
            array(
                'page'          => 'abcgs-ai-bulk-silo',
                'abcgs_message' => rawurlencode( $msg ),
            ),
            admin_url( 'admin.php' )
        );
        wp_safe_redirect( $url );
        exit;
    }

    private function redirect_with_error( $msg ) {
        $url = add_query_arg(
            array(
                'page'        => 'abcgs-ai-bulk-silo',
                'abcgs_error' => rawurlencode( $msg ),
            ),
            admin_url( 'admin.php' )
        );
        wp_safe_redirect( $url );
        exit;
    }
}

new AI_Bulk_ChatGPT_Silo();
